<?php
/**
 * Plugin Name: GTech – Setup Fee for YITH Subscription (Free)
 * Description: Adds a one-time setup fee to YITH subscription products. Fee is charged on the initial order only (per quantity); renewals remain recurring-only. Shows the setup fee in admin Products list and on the frontend price HTML (including Elementor Product Price widget). Also includes shortcodes.
 * Version: 1.1.1
 * Author: GTech
 * Requires at least: 6.0
 * Tested up to: 6.6
 * WC requires at least: 7.0
 * WC tested up to: 9.0
 * Text Domain: gtech-yith-setup-fee
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * WooCommerce HPOS (Custom Order Tables) compatibility declaration
 */
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( '\\Automattic\\WooCommerce\\Utilities\\FeaturesUtil' ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
    }
});

if ( ! class_exists( 'Gtech_YITH_Setup_Fee' ) ) :

class Gtech_YITH_Setup_Fee {

    public function __construct() {
        // Admin: add/save product field
        add_action( 'woocommerce_product_options_general_product_data', [ $this, 'add_field' ] );
        add_action( 'woocommerce_process_product_meta', [ $this, 'save_field' ] );

        // Front: add fee on initial order
        add_action( 'woocommerce_cart_calculate_fees', [ $this, 'add_fee_to_cart' ] );

        // Show fee under line item and persist to order
        add_filter( 'woocommerce_get_item_data', [ $this, 'show_fee_under_item' ], 10, 2 );
        add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'persist_fee_on_order_line_item' ], 10, 4 );

        // Admin products list: append setup fee under Price column
        if ( is_admin() ) {
            add_action( 'manage_product_posts_custom_column', [ $this, 'append_fee_to_price_column' ], 99, 2 );
        }

        // Frontend: append fee under price html (loops, single, Elementor Product Price widget)
        add_filter( 'woocommerce_get_price_html', [ $this, 'append_fee_in_price_html' ], 99, 2 );

        // Shortcodes
        add_shortcode( 'gtech_setup_fee', [ $this, 'shortcode_setup_fee' ] );
        add_shortcode( 'gtech_price_with_setup', [ $this, 'shortcode_price_with_setup' ] );

        // Basic CSS (can be overridden in theme)
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_styles' ] );
    }

    /** Add Setup Fee field in Product > General */
    public function add_field() {
        echo '<div class="options_group show_if_simple show_if_variable">';
        woocommerce_wp_text_input( [
            'id'                => '_gtech_setup_fee',
            'label'             => __( 'Setup fee (one-time)', 'gtech-yith-setup-fee' ),
            'desc_tip'          => true,
            'description'       => __( 'Charged on the initial order only. Multiplies by quantity.', 'gtech-yith-setup-fee' ),
            'type'              => 'number',
            'custom_attributes' => [ 'min' => '0', 'step' => '0.01' ],
        ] );
        echo '</div>';
    }

    /** Save field */
    public function save_field( $post_id ) {
        if ( isset( $_POST['_gtech_setup_fee'] ) ) {
            update_post_meta( $post_id, '_gtech_setup_fee', wc_clean( wp_unslash( $_POST['_gtech_setup_fee'] ) ) );
        }
    }

    /**
     * Add setup fee at checkout (initial order only).
     * Renewal orders created by YITH are generated without the cart context,
     * so this fee is not added to renewals.
     */
    public function add_fee_to_cart( $cart ) {
        if ( is_admin() && ! defined( 'DOING_AJAX' ) ) return;
        if ( empty( $cart->cart_contents ) ) return;

        foreach ( $cart->get_cart() as $key => $item ) {
            if ( empty( $item['data'] ) || ! $item['data'] instanceof WC_Product ) continue;

            $product_id = $item['data']->get_id();
            $setup      = (float) get_post_meta( $product_id, '_gtech_setup_fee', true );

            if ( $setup <= 0 ) continue;

            // Store per-item setup for display and order meta
            $cart->cart_contents[ $key ]['gtech_setup_fee'] = wc_format_decimal( $setup );

            $qty       = max( 1, (int) $item['quantity'] );
            $amount    = $setup * $qty;
            $taxable   = $item['data']->get_tax_status() !== 'none';
            $tax_class = $item['data']->get_tax_class();

            // Add a fee line that honors tax class
            $cart->add_fee( sprintf( '%s – %s', $item['data']->get_name(), __( 'setup fee', 'gtech-yith-setup-fee' ) ), $amount, $taxable, $tax_class );
        }
    }

    /** Show the per-item setup fee under the cart/checkout line */
    public function show_fee_under_item( $item_data, $cart_item ) {
        if ( ! empty( $cart_item['gtech_setup_fee'] ) ) {
            $item_data[] = [
                'name'  => __( 'Setup fee (one-time)', 'gtech-yith-setup-fee' ),
                'value' => wc_price( $cart_item['gtech_setup_fee'] ),
            ];
        }
        return $item_data;
    }

    /** Persist fee info on the order item for admin/customer visibility */
    public function persist_fee_on_order_line_item( $item, $cart_item_key, $values, $order ) {
        if ( ! empty( $values['gtech_setup_fee'] ) ) {
            $item->add_meta_data( __( 'Setup fee (one-time)', 'gtech-yith-setup-fee' ), wc_price( $values['gtech_setup_fee'] ) );
        }
    }

    /** Append Setup Fee info under the Price column on Products list in wp-admin */
    public function append_fee_to_price_column( $column, $post_id ) {
        if ( 'price' !== $column ) return;
        $fee = (float) get_post_meta( $post_id, '_gtech_setup_fee', true );
        if ( $fee > 0 ) {
            echo '<div class="gtech-setup-fee" style="opacity:.75;margin-top:2px;">+' . wc_price( $fee ) . ' <small>' . esc_html__( 'setup (one-time)', 'gtech-yith-setup-fee' ) . '</small></div>';
        }
    }

    /** FRONTEND: append fee in price html so Elementor Product Price shows it too */
    public function append_fee_in_price_html( $price_html, $product ) {
        if ( ! $product instanceof WC_Product ) return $price_html;

        $fee = (float) get_post_meta( $product->get_id(), '_gtech_setup_fee', true );
        if ( $fee <= 0 ) return $price_html;

        $label = esc_html__( 'Setup (one-time)', 'gtech-yith-setup-fee' );
        $html  = sprintf(
            '<div class="gtech-price-with-setup">%s<div class="gtech-setup-line">+ %s <span class="gtech-setup-label">%s</span></div></div>',
            $price_html,
            wc_price( $fee ),
            $label
        );
        return $html;
    }

    /** Shortcode: [gtech_setup_fee product_id="123"] or inside product context without id */
    public function shortcode_setup_fee( $atts ) {
        $atts = shortcode_atts( [ 'product_id' => 0, 'label' => __( 'Setup (one-time)', 'gtech-yith-setup-fee' ) ], $atts, 'gtech_setup_fee' );

        $prod = null;
        if ( ! empty( $atts['product_id'] ) ) {
            $prod = wc_get_product( (int) $atts['product_id'] );
        }
        if ( ! $prod ) {
            global $product;
            if ( $product instanceof WC_Product ) {
                $prod = $product;
            }
        }
        if ( ! $prod ) return '';

        $fee = (float) get_post_meta( $prod->get_id(), '_gtech_setup_fee', true );
        if ( $fee <= 0 ) return '';

        return sprintf( '<div class="gtech-setup-line">+ %s <span class="gtech-setup-label">%s</span></div>', wc_price( $fee ), esc_html( $atts['label'] ) );
    }

    /** Shortcode: [gtech_price_with_setup product_id="123"] */
    public function shortcode_price_with_setup( $atts ) {
        $atts = shortcode_atts( [ 'product_id' => 0 ], $atts, 'gtech_price_with_setup' );

        $prod = null;
        if ( ! empty( $atts['product_id'] ) ) {
            $prod = wc_get_product( (int) $atts['product_id'] );
        }
        if ( ! $prod ) {
            global $product;
            if ( $product instanceof WC_Product ) $prod = $product;
        }
        if ( ! $prod ) return '';

        // Prevent double-append: temporarily remove filter
        remove_filter( 'woocommerce_get_price_html', [ $this, 'append_fee_in_price_html' ], 99 );
        $price_html = $prod->get_price_html();
        add_filter( 'woocommerce_get_price_html', [ $this, 'append_fee_in_price_html' ], 99, 2 );

        return $this->append_fee_in_price_html( $price_html, $prod );
    }

    /** CSS */
    public function enqueue_styles() {
        $css = '
        .gtech-price-with-setup .gtech-setup-line{font-size:0.9em; opacity:.85; margin-top:2px}
        .gtech-price-with-setup .gtech-setup-label{font-size:.85em}
        ';
        wp_register_style( 'gtech-yith-setup-fee', false );
        wp_enqueue_style( 'gtech-yith-setup-fee' );
        wp_add_inline_style( 'gtech-yith-setup-fee', $css );
    }
}

new Gtech_YITH_Setup_Fee();

endif;
