<?php
/**
 * Plugin Name: GTech – Woo Subscription Guard (MU)
 * Description: Prevent duplicate active subscriptions and show notices as toasts (flash messages) globally.
 * Version: 2.0.1
 */

defined('ABSPATH') || exit;

/** === GUARD LOGIC (same as before) === */
add_filter('woocommerce_add_to_cart_validation', function ($passed, $product_id, $quantity, $variation_id = 0) {
    if ( ! is_user_logged_in()
      || ! class_exists('WC_Subscriptions_Product')
      || ! function_exists('wcs_user_has_subscription') ) {
        return $passed;
    }

    $check_id = $variation_id ?: $product_id;
    $is_subscription = is_callable(['WC_Subscriptions_Product','is_subscription'])
        ? WC_Subscriptions_Product::is_subscription($check_id)
        : false;

    if ( ! $is_subscription ) return $passed;

    if ( wcs_user_has_subscription(get_current_user_id(), $check_id, 'active') ) {
        wc_add_notice(__('You already have an active subscription with this product.', 'gtech'), 'error');
        return false;
    }
    return $passed;
}, 10, 4);

/** If Woo redirects after a failed add-to-cart, strip params (no anchor needed). */
add_filter('woocommerce_add_to_cart_redirect', function ($url) {
    if ( function_exists('wc_notice_count') && wc_notice_count('error') > 0 ) {
        $base = wc_get_raw_referer();
        $base = $base ? $base : remove_query_arg(['add-to-cart','quantity','e-redirect']);
        return remove_query_arg(['add-to-cart','quantity','e-redirect'], $base);
    }
    return $url;
});

/** === FOOTER: print (hidden) notices, then convert to toasts === */
add_action('wp_footer', function () {
    if ( ! function_exists('wc_print_notices') ) return;

    // Hidden dump so we can read & clear the queue (prevents future double-renders)
    echo '<div id="gtech-wc-notices-dump" style="display:none">';
    wc_print_notices(); // echoes and CLEARS Woo's notices queue
    echo '</div>';
}, 1);

add_action('wp_footer', function () { ?>
<style>
  /* Hide default Woo notice bars everywhere (we use toasts instead) */
  body .woocommerce-notices-wrapper,
  body .woocommerce-message,
  body .woocommerce-error,
  body .woocommerce-info { display: none !important; }

  /* Toast UI */
  #gtech-toast-wrap
        {
            position: fixed;
            top: 16px;              /* was bottom:16px */
            right: 16px;
            left: auto;
            bottom: auto;           /* unset bottom */
            z-index: 999999;
            display: flex;
            flex-direction: column; /* stack downwards */
            gap: 12px;
            
        }
  .gtech-toast
        {
            min-width:280px;
            max-width:480px;
            padding:12px 14px;
            border-radius:12px;
            box-shadow:0 10px 24px rgba(0,0,0,.18);
            background:#fff;
            color:#111;
            border-left:6px solid #2563eb;
            opacity:0;
            transform:translateY(8px);
            transition:opacity .2s, transform .2s
            
        }
  .gtech-toast.show{opacity:1;transform:translateY(0)}
  .gtech-toast.error{border-color:#dc2626}
  .gtech-toast.success{border-color:#16a34a}
  .gtech-toast.info{border-color:#6b7280}
  .gtech-toast .gtech-close{float:right;cursor:pointer;font-weight:700;margin-left:8px}
</style>

<div id="gtech-toast-wrap" aria-live="polite" aria-atomic="true"></div>

<script>
(function($){
  // De-dupe by plain text (not HTML) across ALL sources
  var seen = new Set();
  function normText(nodeOrHtml){
    var tmp = document.createElement('div');
    if (typeof nodeOrHtml === 'string') tmp.innerHTML = nodeOrHtml;
    else tmp.innerHTML = nodeOrHtml.innerHTML || nodeOrHtml.textContent || '';
    return (tmp.textContent || '').replace(/\s+/g,' ').trim().toLowerCase();
  }

  function toast(type, nodeOrHtml, timeout){
    var text = normText(nodeOrHtml);
    if (!text) return;
    var key = (type||'info') + '|' + text;
    if (seen.has(key)) return;        // already shown from another source
    seen.add(key);

    var host = document.getElementById('gtech-toast-wrap');
    if (!host) return;

    var el = document.createElement('div');
    el.className = 'gtech-toast ' + (type||'info');
    el.innerHTML = '<span class="gtech-close" aria-label="Close">×</span>' + text;
    host.appendChild(el);
    requestAnimationFrame(function(){ el.classList.add('show'); });

    var dismiss = function(){ el.classList.remove('show'); setTimeout(function(){ el.remove(); }, 200); };
    el.querySelector('.gtech-close').addEventListener('click', dismiss);
    setTimeout(dismiss, timeout || 5000);
  }

  function processWrapper(wrap){
    if (wrap.dataset.gtechProcessed) return; // guard against reprocessing
    wrap.dataset.gtechProcessed = '1';

    var type = wrap.classList.contains('woocommerce-error') ? 'error' :
               wrap.classList.contains('woocommerce-message') ? 'success' : 'info';

    var lis = wrap.querySelectorAll('li');
    if (lis.length) lis.forEach(function(li){ toast(type, li); });
    else toast(type, wrap);
  }

  function collect(){
    // 1) Hidden dump we printed server-side (clears the queue)
    var dump = document.getElementById('gtech-wc-notices-dump');
    if (dump){
      dump.querySelectorAll('.woocommerce-error, .woocommerce-message, .woocommerce-info').forEach(processWrapper);
      dump.innerHTML = '';
    }

    // 2) Any other wrappers on the page (header, mini-cart, fragments, etc.)
    document.querySelectorAll('.woocommerce-error, .woocommerce-message, .woocommerce-info')
      .forEach(function(wrap){
        processWrapper(wrap);
        // If it lives in the header notice area, remove it so only the toast is visible
        if (wrap.closest('header, .site-header, .woocommerce-notices-wrapper')) wrap.remove();
      });
  }

  document.addEventListener('DOMContentLoaded', collect);
  // Cover all Woo AJAX flows that may emit notices
  $(document.body).on(
    'added_to_cart wc_fragments_loaded wc_fragments_refreshed removed_from_cart updated_wc_div updated_cart_totals wc_cart_emptied',
    collect
  );
})(jQuery);
</script>

<?php }, 99);
